-- File: scripts/events/ConsumeObjectsEvent.lua
--[[ 
Mod: Take Single Objects

Author: Lism
Date: 2025-09-13
Version: 1.0.0.0

Changelog:
    v 1.0.0.0 @2025-09-13 - Initial release
--]]

ConsumeObjectsEvent = {}
local ConsumeObjectsEvent_mt = Class(ConsumeObjectsEvent, Event)
InitEventClass(ConsumeObjectsEvent, "ConsumeObjectsEvent")

local EPS = 0.001
local DBG = false

local function dlog(fmt, ...) if DBG then Logging.info("[TakeSingleObjects] " .. string.format(fmt, ...)) end end

local function normPath(p) if p == nil then return "" end p = string.gsub(p, "\\", "/"); return string.lower(p) end
local function objectConfigPath(obj) local cfg = obj and (obj.configFileName or obj.configFile) or ""; return normPath(cfg) end
local function endsWith(str, suffix) if str == nil or suffix == nil then return false end local ls, le = #str, #suffix if le > ls then return false end return string.sub(str, ls-le+1, ls) == suffix end

local function matchDefForObject(obj)
    if TakeSingleObjects == nil or TakeSingleObjects.registry == nil then return nil end
    local cfg = objectConfigPath(obj)
    for _, def in pairs(TakeSingleObjects.registry) do
        local e = def.__ends or (def.match and def.match.endsWith)
        if e ~= nil and e ~= "" and endsWith(cfg, e) then return def end
    end
    return nil
end

function ConsumeObjectsEvent.emptyNew() return Event.new(ConsumeObjectsEvent_mt) end
function ConsumeObjectsEvent.new(srcNetId, fu, fillTypeIndex, objectIds)
    local self = ConsumeObjectsEvent.emptyNew()
    self.srcNetId = srcNetId or 0
    self.fu = fu or 1
    self.fillTypeIndex = fillTypeIndex or 0
    self.objectIds = (type(objectIds) == "table") and objectIds or {}
    return self
end

function ConsumeObjectsEvent:readStream(streamId, connection)
    self.srcNetId = NetworkUtil.readNodeObjectId(streamId)
    self.fu = streamReadUInt8(streamId)
    self.fillTypeIndex = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)
    local n = streamReadUInt8(streamId)
    self.objectIds = {}
    for i=1, n do self.objectIds[i] = NetworkUtil.readNodeObjectId(streamId) end
    self:run(connection)
end

function ConsumeObjectsEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObjectId(streamId, self.srcNetId)
    streamWriteUInt8(streamId, self.fu or 1)
    streamWriteUIntN(streamId, self.fillTypeIndex or 0, FillTypeManager.SEND_NUM_BITS)
    local n = math.min(255, (type(self.objectIds) == "table" and #self.objectIds) or 0)
    streamWriteUInt8(streamId, n)
    for i=1, n do NetworkUtil.writeNodeObjectId(streamId, self.objectIds[i]) end
end

local function getVehicleFillInfo(v, ftIdx)
    local liters, fu = 0, nil
    if v ~= nil and v.getFillUnits ~= nil then
        for _, fuData in ipairs(v:getFillUnits() or {}) do
            local i = fuData.fillUnitIndex or fuData.index or 1
            local lvl = fuData.fillLevel or 0
            if lvl > EPS and (v.getFillUnitSupportsFillType == nil or v:getFillUnitSupportsFillType(i, ftIdx)) then
                liters = liters + lvl
                fu = fu or i
            end
        end
    end
    return liters, fu
end

local function getConsumableTypeByFU(obj, fu)
    local spec = obj and obj.spec_consumable
    if spec and spec.types then
        for i, t in ipairs(spec.types) do
            if t.fillUnitIndex == fu then return t, i end
        end
    end
    return nil, nil
end

local function findHighestEmptySlotIndex(typeObj)
    if typeObj == nil or typeObj.storageSlots == nil then return nil end
    for i = #typeObj.storageSlots, 1, -1 do
        local s = typeObj.storageSlots[i]
        if (s.consumableVariationIndex or 0) == 0 then return i end
    end
    return nil
end

local function getVariationIndexFromItem(v, objectFu)
    -- 1) aus Store‑Config (Name/SaveId) -> globaler Index
    if ConfigurationUtil ~= nil and g_storeManager ~= nil and v ~= nil then
        local vXml = v.configFileName or v.configFile
        if vXml ~= nil and v.configurations ~= nil then
            local idx = v.configurations["consumableName"]
            if idx ~= nil then
                local cfgItem = ConfigurationUtil.getConfigItemByConfigId(vXml, "consumableName", idx)
                local saveId = cfgItem and cfgItem.saveId
                if saveId ~= nil and g_consumableManager ~= nil and g_consumableManager.variationsByName ~= nil then
                    local var = g_consumableManager.variationsByName[string.upper(saveId)]
                    if var ~= nil and var.index ~= nil then return var.index end
                end
            end
        end
    end
    -- 2) Fallback: direkt vom Einzelteil (falls selbst Consumable)
    if v ~= nil and v.getConsumableVariationIndexByFillUnitIndex ~= nil then
        local ok, idx = pcall(function() return v:getConsumableVariationIndexByFillUnitIndex(objectFu or 1) end)
        if ok and type(idx) == "number" and idx > 0 then return idx end
    end
    -- 3) Default
    return 1
end

local function refreshPalletAfterChange(pallet, typeName)
    if pallet == nil then return end
    if pallet.updateConsumable ~= nil and typeName ~= nil then pcall(function() pallet:updateConsumable(typeName, 0, false, true) end) end
    if pallet.setPalletTensionBeltNodesDirty ~= nil then pcall(function() pallet:setPalletTensionBeltNodesDirty() end) end
    if pallet.updatePalletStraps ~= nil then pcall(function() pallet:updatePalletStraps() end) end
end

function ConsumeObjectsEvent:run(connection)
    if g_currentMission == nil or not g_currentMission:getIsServer() then dlog("ConsumeObjectsEvent:run on client -> ignore"); return end

    local srcObj = NetworkUtil.getObject(self.srcNetId)
    if srcObj == nil then return end

    local def = matchDefForObject(srcObj)
    if def == nil then dlog("No mapping for source -> skip"); return end

    local ft = self.fillTypeIndex or (def.object and def.object.fillTypeIndex)
    local fu = self.fu or 1

    local farmId = 1
    if srcObj.getOwnerFarmId ~= nil then
        local owner = srcObj:getOwnerFarmId() or 0
        if owner ~= 0 then farmId = owner end
    end

    local wantXml = normPath(def.object and def.object.xml)

    for i=1, ((type(self.objectIds) == "table" and #self.objectIds) or 0) do
        local id = self.objectIds[i]
        local v = NetworkUtil.getObject(id)
        if v ~= nil and not v.isDeleted then
            repeat
                if wantXml ~= nil and wantXml ~= "" then if not endsWith(objectConfigPath(v), wantXml) then break end end
                -- nur Items der gleichen Owner‑Farm akzeptieren (gleiche Herkunft beim zurücklegen)
                if v.getOwnerFarmId ~= nil then local vf = v:getOwnerFarmId() or 0; if vf ~= 0 and vf ~= farmId then break end end

                local liters, objectFu = getVehicleFillInfo(v, ft)
                if liters <= EPS then break end

                if def.object.isConsumable and srcObj.spec_consumable ~= nil then
                    local typeObj, typeIndex = getConsumableTypeByFU(srcObj, fu)
                    if typeObj == nil then break end

                    local units = math.floor(liters + 0.0001)
                    local added = 0

                    for n=1, units do
                        local slotIndex = findHighestEmptySlotIndex(typeObj)
                        if slotIndex == nil then break end -- voll
                        local varIdx = getVariationIndexFromItem(v, objectFu)
                        local ok1 = pcall(function() return srcObj:setConsumableSlotVariationIndex(typeIndex or 1, slotIndex, varIdx) end)
                        if not ok1 then break end
                        added = added + 1
                    end

                    if added > 0 then
                        local okAdd, _ = pcall(function() return srcObj:addFillUnitFillLevel(farmId, fu, added, ft, ToolType.UNDEFINED, nil) end)
                        if okAdd then
                            local rest = math.max(0, units - added)
                            if rest <= EPS then
                                if v.delete ~= nil then v:delete() end
                            else
                                if objectFu ~= nil then
                                    pcall(function() v:setFillUnitFillLevel(objectFu, rest, ft, false) end)
                                    if v.updateFillUnitFillLevel ~= nil then pcall(function() v:updateFillUnitFillLevel(objectFu, rest) end) end
                                end
                            end
                            refreshPalletAfterChange(srcObj, typeObj and typeObj.typeName)
                        end
                    end
                else
                    -- Standard: klassische FillUnits
                    local added = 0
                    if srcObj.addFillUnitFillLevel ~= nil then
                        local ok, ret = pcall(function() return srcObj:addFillUnitFillLevel(farmId, fu, liters, ft, ToolType.UNDEFINED, nil) or 0 end)
                        if ok then added = math.max(0, ret or 0) end
                    end
                    local rest = math.max(0, liters - (added or 0))
                    if rest <= EPS then
                        if v.delete ~= nil then v:delete() end
                    else
                        local _, objectFu2 = getVehicleFillInfo(v, ft)
                        if objectFu2 ~= nil then
                            pcall(function() v:setFillUnitFillLevel(objectFu2, rest, ft, false) end)
                            if v.updateFillUnitFillLevel ~= nil then pcall(function() v:updateFillUnitFillLevel(objectFu2, rest) end) end
                        end
                    end
                end
            until true
        end
    end

    if g_server ~= nil and connection ~= nil then g_server:broadcastEvent(self, false, connection) end
end
