--[[
    BaggingMachine

    Specialization for bagging machines that can create boxes with fill types.

        @author:    [LSFM] FarmerAndy, [BG] BayernGamers
        @date:      06.03.2025
        @version:   1.2

        History:    v1.0 @29.08.2024 - initial implementation in FS22
                    -------------------------------------------------------------------------------------------
					v1.1 @02.09.2024 - added support for consumables
					-------------------------------------------------------------------------------------------
					v1.2 @06.03.2025 - converted to FS25
					-------------------------------------------------------------------------------------------

		License:    Terms:
						Usage:
							Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage:
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]

source("dataS/scripts/vehicles/specializations/events/ReceivingHopperSetCreateBoxesEvent.lua")

BaggingMachine = {}
BaggingMachine.MOD_NAME = g_currentModName

function BaggingMachine.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(FillUnit, specializations) and SpecializationUtil.hasSpecialization(Dischargeable, specializations)
end

function BaggingMachine.initSpecialization()
	local schema = Vehicle.xmlSchema

	schema:setXMLSpecializationType("BaggingMachine")
	schema:register(XMLValueType.INT, "vehicle.baggingMachine#fillUnitIndex", "Fill unit index", 1)
	schema:register(XMLValueType.FLOAT, "vehicle.baggingMachine#spawnNodeYOffset", "Y offset for Pallet spawn Node", 0)
	schema:register(XMLValueType.INT, "vehicle.baggingMachine#consumableFillUnitIndex", "Consumable FillUnit Index", 2)
	schema:register(XMLValueType.FLOAT, "vehicle.baggingMachine#consumedAmount", "Amount consumed at the bagging process", 1)
	schema:register(XMLValueType.STRING, "vehicle.baggingMachine#refillConsumablesWarning", "Refill supply warning text")

	schema:register(XMLValueType.STRING, "vehicle.baggingMachine#dependsOnAnimation", "Needs Animation xy for enabling spawner", "false")
	schema:register(XMLValueType.NODE_INDEX, "vehicle.baggingMachine.boxes#spawnPlaceNode", "Spawn place node")
	schema:register(XMLValueType.STRING, "vehicle.baggingMachine.boxes.box(?)#fillType", "Fill type name")
	schema:register(XMLValueType.STRING, "vehicle.baggingMachine.boxes.box(?)#filename", "Box filename")
	schema:setXMLSpecializationType()

	local schemaSavegame = Vehicle.xmlSchemaSavegame
	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?)." .. BaggingMachine.MOD_NAME .. ".baggingMachine#createBoxes", "Create boxes")
end

function BaggingMachine.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "setCreateBoxes", BaggingMachine.setCreateBoxes)
	SpecializationUtil.registerFunction(vehicleType, "getCanSpawnNextBox", BaggingMachine.getCanSpawnNextBox)
	SpecializationUtil.registerFunction(vehicleType, "collisionTestCallback", BaggingMachine.collisionTestCallback)
	SpecializationUtil.registerFunction(vehicleType, "createBox", BaggingMachine.createBox)
	SpecializationUtil.registerFunction(vehicleType, "onCreateBoxFinished", BaggingMachine.onCreateBoxFinished)

	SpecializationUtil.registerFunction(vehicleType, "getRefillNeededWarning", BaggingMachine.getRefillNeededWarning)
end

function BaggingMachine.registerOverwrittenFunctions(vehicleType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "handleDischargeRaycast", BaggingMachine.handleDischargeRaycast)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCanBeSelected", BaggingMachine.getCanBeSelected)
end

function BaggingMachine.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onLoad", BaggingMachine)
	SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", BaggingMachine)
	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", BaggingMachine)
	SpecializationUtil.registerEventListener(vehicleType, "onFillUnitFillLevelChanged", BaggingMachine)
end

function BaggingMachine:onLoad(savegame)
	self.spec_baggingMachine = {}
	local spec = self.spec_baggingMachine

	--print("InputAction: " .. tostring(InputAction))
	--print("InputAction.IMPLEMENT_EXTRA: " .. tostring(InputAction.IMPLEMENT_EXTRA))

	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine#unloadingDelay", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine#unloadInfoIndex", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine#dischargeInfoIndex", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.tipTrigger#index", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.boxTrigger#index", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.fillScrollerNodes.fillScrollerNode", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.fillEffect", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.fillEffect", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.boxTrigger#litersPerMinute", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.raycastNode#index", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.raycastNode#raycastLength", "Dischargeable functionalities")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.boxTrigger#boxSpawnPlaceIndex", "vehicle.baggingMachine.boxes#spawnPlaceNode")
	XMLUtil.checkDeprecatedXMLElements(self.xmlFile, "vehicle.baggingMachine.boxTrigger.box(0)", "vehicle.baggingMachine.boxes.box(0)")

	spec.fillUnitIndex = self.xmlFile:getValue("vehicle.baggingMachine#fillUnitIndex", 1)
	spec.consumableFillUnitIndex = self.xmlFile:getValue("vehicle.baggingMachine#consumableFillUnitIndex", 2)
	spec.consumedAmount = self.xmlFile:getValue("vehicle.baggingMachine#consumedAmount", 1)
	spec.refillConsumablesWarning = self.xmlFile:getValue("vehicle.baggingMachine#refillConsumablesWarning")

	spec.spawnNodeYOffset = self.xmlFile:getValue("vehicle.baggingMachine#spawnNodeYOffset", 0)
	spec.dependsOnAnimation = self.xmlFile:getValue("vehicle.baggingMachine#dependsOnAnimation", "false")
	spec.spawnPlace = self.xmlFile:getValue("vehicle.baggingMachine.boxes#spawnPlaceNode", nil, self.components, self.i3dMappings)
	spec.boxes = {}
	spec.actionEvents = {}
	local i = 0

	while true do
		local baseName = string.format("vehicle.baggingMachine.boxes.box(%d)", i)

		if not self.xmlFile:hasProperty(baseName) then
			break
		end

		local fillTypeStr = self.xmlFile:getValue(baseName .. "#fillType")
		local filename = self.xmlFile:getValue(baseName .. "#filename")
		local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeStr)

		if fillTypeIndex ~= nil then
			spec.boxes[fillTypeIndex] = filename
		else
			Logging.xmlWarning(self.xmlFile, "Invalid fillType '%s'", fillTypeStr)
		end

		i = i + 1
	end

	spec.createBoxes = false
	spec.lastBox = nil
	spec.creatingBox = false

	if savegame ~= nil then
		spec.createBoxes = savegame.xmlFile:getValue(savegame.key .. "." .. BaggingMachine.MOD_NAME .. ".baggingMachine#createBoxes", spec.createBoxes)
	end

	if not self.isServer then
		SpecializationUtil.removeEventListener(self, "onUpdateTick", BaggingMachine)
	end
end

function BaggingMachine:saveToXMLFile(xmlFile, key, usedModNames)
	local spec = self.spec_baggingMachine

	xmlFile:setValue(key .. "#createBoxes", spec.createBoxes)
end

function BaggingMachine:onUpdateTick(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local spec = self.spec_baggingMachine

	if spec.createBoxes and self:getDischargeState() == Dischargeable.DISCHARGE_STATE_OFF and self:getCanSpawnNextBox() then
		self:createBox()
	end
end

function BaggingMachine:setCreateBoxes(state, noEventSend)
	local spec = self.spec_baggingMachine

	if state and self:getFillUnitFillLevel(spec.consumableFillUnitIndex) == 0 then
		state = false
	end

	if state ~= spec.createBoxes then
		ReceivingHopperSetCreateBoxesEvent.sendEvent(self, state, noEventSend)

		spec.createBoxes = state

		BaggingMachine.updateActionEvents(self)

		spec.lastBox = nil
	end
end

function BaggingMachine:getCanSpawnNextBox()
	local spec = self.spec_baggingMachine

	if self:getFillUnitFillLevel(spec.consumableFillUnitIndex) == 0 then
		--print("Returning false because consumable fill level == 0")

		local warning = self:getRefillNeededWarning()
		--print("CanSpawnNextBox? Warning: " .. tostring(warning))
		g_currentMission:showBlinkingWarning(warning, 5000)

		self:setCreateBoxes(false)
		return false
	end

	if spec.dependsOnAnimation ~= nil and spec.dependsOnAnimation ~= "false" then
		local animation = self:getAnimationByName(spec.dependsOnAnimation)

		if animation ~= nil then
			local isAnimationPlaying = self:getIsAnimationPlaying(spec.dependsOnAnimation)
			local animationTime = self:getAnimationTime(spec.dependsOnAnimation)
			local stopTime = animation.stopTime

			--print("Animation: " .. tostring(spec.dependsOnAnimation))
			--print("Is playing: " .. tostring(isAnimationPlaying))
			--print("Animation time: " .. tostring(animationTime))

			if animationTime ~= 0 then
				if spec.lastBox ~= nil and spec.lastBox:getFillUnitFreeCapacity(1) > 0 then
					spec.lastBox = nil
				end

				--print("Returning false because animation time ~= 0")
				return false
			end
		end
	end

	if spec.creatingBox then
		--print("Returning false because creating box")
		return false
	end

	local fillType = self:getFillUnitFillType(spec.fillUnitIndex)

	if spec.boxes[fillType] ~= nil then
		if spec.lastBox ~= nil and spec.lastBox:getFillUnitFreeCapacity(1) > 0 then
			--print("Returning false because last box has free capacity")
			return false
		end

		local xmlFilename = Utils.getFilename(spec.boxes[fillType], self.baseDirectory)
		local size = StoreItemUtil.getSizeValues(xmlFilename, "vehicle", 0)
		local x, y, z = getWorldTranslation(spec.spawnPlace)
		local rx, ry, rz = getWorldRotation(spec.spawnPlace)
		spec.foundObjectAtSpawnPlace = false

		overlapBox(x, y, z, rx, ry, rz, size.width * 0.5, 2, size.length * 0.5, "collisionTestCallback", self, 5468288)

		--print("Returning" .. tostring(not spec.foundObjectAtSpawnPlace) .. " because found object at spawn place")
		return not spec.foundObjectAtSpawnPlace
	end

	--print("Returning false because no box for fill type")
	return false
end

function BaggingMachine:collisionTestCallback(transformId)
	if (g_currentMission.nodeToObject[transformId] ~= nil or g_currentMission.players[transformId] ~= nil) and g_currentMission.nodeToObject[transformId] ~= self then
		local spec = self.spec_baggingMachine
		spec.foundObjectAtSpawnPlace = true
	end
end

function BaggingMachine:createBox()
    local spec = self.spec_baggingMachine
	
    if self.isServer then
        if spec.createBoxes then
            local fillType = self:getFillUnitFillType(spec.fillUnitIndex)
            if spec.boxes[fillType] ~= nil then
                local x, y, z = getWorldTranslation(spec.spawnPlace)
                local rx, ry, rz = getWorldRotation(spec.spawnPlace)
                local xmlFilename = Utils.getFilename(spec.boxes[fillType], self.baseDirectory)

                spec.creatingBox = true

				if spec.spawnNodeYOffset ~= nil and spec.spawnNodeYOffset ~= 0 then
					y = y + spec.spawnNodeYOffset
				end

                local data = VehicleLoadingData.new()
                data:setFilename(xmlFilename)
                data:setPosition(x, y, z)
                data:setRotation(rx, ry, rz)
                data:setPropertyState(VehiclePropertyState.OWNED)
                data:setOwnerFarmId(self:getOwnerFarmId())

                data:load(self.onCreateBoxFinished, self, nil)
            end
        end
    end
end

function BaggingMachine:onCreateBoxFinished(vehicles, vehicleLoadState, arguments)
    local spec = self.spec_baggingMachine
	local consumableFillType = self:getFillUnitFillType(spec.consumableFillUnitIndex)
    spec.creatingBox = false

    if vehicleLoadState == VehicleLoadingState.OK then
        spec.lastBox = vehicles[1]
		self:addFillUnitFillLevel(self:getOwnerFarmId(), spec.consumableFillUnitIndex, 0 - spec.consumedAmount, consumableFillType, ToolType.UNDEFINED, nil)
    end
end

function BaggingMachine:handleDischargeRaycast(superFunc, dischargeNode, hitObject, hitShape, hitDistance, hitFillUnitIndex, hitTerrain)
	local stopDischarge = false

	if hitObject ~= nil then
		local fillType = self:getDischargeFillType(dischargeNode)
		local allowFillType = hitObject:getFillUnitAllowsFillType(hitFillUnitIndex, fillType)

		if allowFillType and hitObject:getFillUnitFreeCapacity(hitFillUnitIndex) > 0 then
			self:setDischargeState(Dischargeable.DISCHARGE_STATE_OBJECT, true)
		else
			stopDischarge = true
		end
	else
		stopDischarge = true
	end

	if stopDischarge and self:getDischargeState() == Dischargeable.DISCHARGE_STATE_OBJECT then
		self:setDischargeState(Dischargeable.DISCHARGE_STATE_OFF, true)
	end
end

function BaggingMachine:getCanBeSelected(superFunc)
	return true
end

function BaggingMachine:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	if self.isClient then
		local spec = self.spec_baggingMachine

		self:clearActionEventsTable(spec.actionEvents)

		if isActiveForInputIgnoreSelection then
			local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.IMPLEMENT_EXTRA, self, BaggingMachine.actionEventToggleBoxCreation, false, true, false, true, nil)

			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
			BaggingMachine.updateActionEvents(self)
		end
	end
end

function BaggingMachine:onFillUnitFillLevelChanged(fillUnitIndex, fillLevelDelta, fillType, toolType, fillPositionData, appliedDelta)
	if self:getFillUnitFillLevel(fillUnitIndex) > 0 then
		self:raiseActive()
	end
end

function BaggingMachine:actionEventToggleBoxCreation(actionName, inputValue, callbackState, isAnalog)
	local spec = self.spec_baggingMachine
	local consumableFillLevel = self:getFillUnitFillLevel(spec.consumableFillUnitIndex)

	--print("spec.createBoxes: " .. tostring(spec.createBoxes))
	if consumableFillLevel == 0 and spec.createBoxes == false then
		--print("Consumable fill level == 0")

		local warning = self:getRefillNeededWarning()
		--print("Warning: " .. tostring(warning))
		g_currentMission:showBlinkingWarning(warning, 5000)
	end

	self:setCreateBoxes(not spec.createBoxes)
end

function BaggingMachine:updateActionEvents()
	local spec = self.spec_baggingMachine
	local actionEvent = spec.actionEvents[InputAction.IMPLEMENT_EXTRA]

	if actionEvent ~= nil then
		if spec.createBoxes then
			g_inputBinding:setActionEventText(actionEvent.actionEventId, g_i18n:getText("action_disableBagSpawning"))
		else
			g_inputBinding:setActionEventText(actionEvent.actionEventId, g_i18n:getText("action_enableBagSpawning"))
		end
	end
end

function BaggingMachine:getRefillNeededWarning()
	--print("Getting refill needed warning")
	local spec = self.spec_baggingMachine
    local text = g_i18n:getText(spec.refillConsumablesWarning or "warning_notAcceptedHere")
	local fillLevel = self:getFillUnitFillLevel(spec.consumableFillUnitIndex)

	--print("Text: " .. tostring(text))

	if fillLevel == 0 then
		--print("Returning warning because fill level == 0")
		return text
	end
end

